<?php
/**
 * avaliar_funcionario.php — Painel de avaliação de desempenho (Direção)
 *
 * Requisitos:
 * - PHP 7.4+
 * - MySQL/MariaDB (mysqli)
 * - conexao.php com $conn (mysqli)
 * - Sessão com $_SESSION['usuario']['perfil'] e ['nome']
 *
 * Segurança e Robustez:
 * - Verificação de perfil (Direção/Admin/Subdireções)
 * - CSRF token
 * - Prepared statements
 * - Estados: RASCUNHO e FINAL (final bloqueia edição)
 * - Auto-criação das tabelas e critérios padrão (se vazias)
 * - Exportação CSV do histórico
 * - Autosave silencioso
 */

session_start();
require_once __DIR__ . '/conexao.php';

// ======= CONTROLE DE ACESSO REFORÇADO =======
$perfil = strtoupper((string)($_SESSION['usuario']['perfil'] ?? ''));
$nomeUsuario = (string)($_SESSION['usuario']['nome'] ?? '');
$avaliadorId = $_SESSION['usuario']['id'] ?? null;

// ✅ Permite múltiplos perfis
if (!in_array($perfil, ['ADMINISTRADOR', 'SUBDIRECCAO PEDAGOGICA', 'DIRECCAO'])) {
    header('HTTP/1.1 403 Forbidden');
    echo '<h2>Acesso negado</h2><p>Este painel é exclusivo para a Direção e Subdireção Pedagógica.</p>';
    exit;
}

// ======= CSRF =======
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}
$csrf = $_SESSION['csrf_token'];

// ======= AUTO-CRIAÇÃO DAS TABELAS =======
$conn->query("CREATE TABLE IF NOT EXISTS criterios_avaliacao (
    id INT AUTO_INCREMENT PRIMARY KEY,
    nome VARCHAR(150) NOT NULL,
    descricao TEXT NULL,
    categoria ENUM('Pedagógico','Administrativo','Comportamental','Outro') NOT NULL DEFAULT 'Outro',
    peso DECIMAL(5,2) NOT NULL DEFAULT 1.00,
    ativo TINYINT(1) NOT NULL DEFAULT 1,
    ordem INT NOT NULL DEFAULT 0
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;");

$conn->query("CREATE TABLE IF NOT EXISTS avaliacoes (
    id INT AUTO_INCREMENT PRIMARY KEY,
    usuario_id INT NOT NULL,
    periodo VARCHAR(60) NOT NULL,
    data_avaliacao DATE NOT NULL,
    estado ENUM('RASCUNHO','FINAL') NOT NULL DEFAULT 'RASCUNHO',
    total_20 DECIMAL(6,2) NOT NULL DEFAULT 0.00,
    observacoes TEXT NULL,
    plano_melhoria TEXT NULL,
    avaliado_por VARCHAR(150) NULL,
    avaliador_id INT NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (usuario_id) REFERENCES usuarios(id),
    FOREIGN KEY (avaliador_id) REFERENCES usuarios(id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;");

$conn->query("CREATE TABLE IF NOT EXISTS avaliacao_detalhes (
    id INT AUTO_INCREMENT PRIMARY KEY,
    avaliacao_id INT NOT NULL,
    criterio_id INT NOT NULL,
    nota5 DECIMAL(4,2) NOT NULL DEFAULT 0.0, -- escala 0 a 5 por critério
    comentario TEXT NULL,
    FOREIGN KEY (avaliacao_id) REFERENCES avaliacoes(id) ON DELETE CASCADE,
    FOREIGN KEY (criterio_id) REFERENCES criterios_avaliacao(id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;");


// Inserir critérios padrão se estiver vazio
$rc = $conn->query("SELECT COUNT(*) c FROM criterios_avaliacao");
if ($rc && ($r = $rc->fetch_assoc()) && (int)$r['c'] === 0) {
    $padrao = [
        ['Planejamento das aulas', 'Planos claros, alinhados a objetivos e conteúdos.', 'Pedagógico', 1.50, 1, 1],
        ['Domínio do conteúdo', 'Conhecimento seguro e atualizado do componente.', 'Pedagógico', 1.50, 1, 2],
        ['Metodologia e recursos', 'Uso de estratégias variadas e recursos/TIC.', 'Pedagógico', 1.20, 1, 3],
        ['Gestão de sala', 'Clima, disciplina, participação e motivação.', 'Pedagógico', 1.20, 1, 4],
        ['Avaliação da aprendizagem', 'Critérios, instrumentos e feedback formativo.', 'Pedagógico', 1.20, 1, 5],
        ['Inovação pedagógica', 'Metodologias ativas e projetos.', 'Pedagógico', 0.90, 1, 6],
        ['Assiduidade e pontualidade', 'Cumprimento de horários e presença.', 'Administrativo', 1.40, 1, 7],
        ['Cumprimento de prazos e normas', 'Entrega de documentos e respeito às normas.', 'Administrativo', 1.30, 1, 8],
        ['Relacionamento interpessoal', 'Colaboração com direção, colegas e alunos.', 'Comportamental', 1.00, 1, 9],
        ['Ética e postura', 'Responsabilidade, respeito e confidencialidade.', 'Comportamental', 1.20, 1, 10],
        ['Formação contínua', 'Participa em ações formativas.', 'Administrativo', 0.80, 1, 11],
        ['Pontualidade na Entrega de Planos de Aula', 'Avalia a consistência na entrega dos planos de aula por mês.', 'Pedagógico', 1.50, 1, 12],
        ['Atraso no Lançamento de Notas', 'Penalidade aplicada automaticamente por não ter lançado as notas a tempo.', 'Outro', 2.50, 1, 13]
    ];
    $stmt = $conn->prepare("INSERT INTO criterios_avaliacao (nome,descricao,categoria,peso,ativo,ordem) VALUES (?,?,?,?,?,?)");
    foreach ($padrao as $c) {
        $stmt->bind_param('sssdis', $c[0], $c[1], $c[2], $c[3], $c[4], $c[5]);
        $stmt->execute();
    }
    $stmt->close();
}


// ======= FUNÇÕES AUXILIARES =======
function all($conn, $sql, $types = null, ...$params) {
    if ($types) {
        $stmt = $conn->prepare($sql);
        if ($stmt === false) { return []; }
        $stmt->bind_param($types, ...$params);
        $stmt->execute();
        $res = $stmt->get_result();
        $stmt->close();
    } else {
        $res = $conn->query($sql);
    }
    $o = [];
    if ($res) {
        while ($r = $res->fetch_assoc()) {
            $o[] = $r;
        }
    }
    return $o;
}
function h($s) {
    return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8');
}
function getUserNameById($usuarios, $id) {
    foreach ($usuarios as $u) {
        if ((int)$u['id'] === (int)$id) {
            return $u['nome'];
        }
    }
    return 'N/A';
}

// Carregar listas base
$usuarios = all($conn, "SELECT id, nome, perfil FROM usuarios ORDER BY nome");
$criterios = all($conn, "SELECT * FROM criterios_avaliacao WHERE ativo=1 ORDER BY categoria, ordem, nome");

// ======= POST HANDLERS =======
$errors = [];
$okMsg = '';
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!hash_equals($csrf, $_POST['csrf_token'] ?? '')) {
        $errors[] = 'Token CSRF inválido. Tente novamente.';
    } else {
        $acao = $_POST['acao'] ?? '';

        if ($acao === 'export_csv') {
            $f_user = (int)($_POST['f_user'] ?? 0);
            $f_estado = $_POST['f_estado'] ?? '';
            $f_periodo = $_POST['f_periodo'] ?? '';
            $w = [];
            if ($f_user) $w[] = 'a.usuario_id=' . $f_user;
            if ($f_estado !== '') $w[] = "a.estado='" . $conn->real_escape_string($f_estado) . "'";
            if ($f_periodo !== '') $w[] = "a.periodo='" . $conn->real_escape_string($f_periodo) . "'";
            $where = $w ? ('WHERE ' . implode(' AND ', $w)) : '';
            $sql = "SELECT a.id, u.nome AS funcionario, u.perfil, a.periodo, a.data_avaliacao, a.estado, a.total_20
                     FROM avaliacoes a JOIN usuarios u ON u.id=a.usuario_id $where ORDER BY a.created_at DESC";
            $rows = all($conn, $sql);
            header('Content-Type: text/csv; charset=utf-8');
            header('Content-Disposition: attachment; filename=avaliacoes_funcionarios.csv');
            $out = fopen('php://output', 'w');
            fputcsv($out, ['ID', 'Funcionário', 'Perfil', 'Período', 'Data', 'Estado', 'Total (0-20)']);
            foreach ($rows as $r) {
                fputcsv($out, [$r['id'], $r['funcionario'], $r['perfil'], $r['periodo'], $r['data_avaliacao'], $r['estado'], $r['total_20']]);
            }
            fclose($out);
            exit;
        }

        // Salvar/Finalizar
        $avaliacao_id = (int)($_POST['avaliacao_id'] ?? 0);
        $usuario_id = (int)($_POST['usuario_id'] ?? 0);
        $periodo = trim($_POST['periodo'] ?? '');
        $data_avaliacao = trim($_POST['data_avaliacao'] ?? '');
        $observacoes = trim($_POST['observacoes'] ?? '');
        $plano = trim($_POST['plano_melhoria'] ?? '');
        $estado = ($acao === 'finalizar') ? 'FINAL' : 'RASCUNHO';

        if (!$usuario_id) $errors[] = 'Selecione o funcionário.';
        if ($periodo === '') $errors[] = 'Informe o período (ex.: 1º Trimestre 2025).';
        if ($data_avaliacao === '') $errors[] = 'Informe a data da avaliação.';

        // CÁLCULO REFORÇADO: NOTA TOTAL (0–20)
        $itens = $_POST['itens'] ?? [];
        $sumW = 0;
        $sumWN = 0;
        foreach ($criterios as $c) {
            $cid = (int)$c['id'];
            $peso = (float)$c['peso'];
            $nota5 = isset($itens[$cid]['nota5']) ? (float)$itens[$cid]['nota5'] : 0.0;
            $sumW += $peso;
            $sumWN += $peso * $nota5;
        }
        $total20 = $sumW > 0 ? round(($sumWN / $sumW) * 4, 2) : 0.00;

        if (!$errors) {
            $isUpdate = ($avaliacao_id > 0);
            $query = "";
            $nomeFuncionario = getUserNameById($usuarios, $usuario_id);

            if ($isUpdate) {
                // Bloqueio se já finalizada
                $chk = $conn->prepare("SELECT estado FROM avaliacoes WHERE id=?");
                $chk->bind_param('i', $avaliacao_id);
                $chk->execute();
                $cur = $chk->get_result()->fetch_assoc();
                $chk->close();
                if ($cur && $cur['estado'] === 'FINAL') {
                    $errors[] = 'Esta avaliação já foi finalizada e não pode ser editada.';
                } else {
                    $query = "UPDATE avaliacoes SET usuario_id=?, periodo=?, data_avaliacao=?, estado=?, total_20=?, observacoes=?, plano_melhoria=?, avaliado_por=?, avaliador_id=? WHERE id=?";
                }
            } else {
                $query = "INSERT INTO avaliacoes (usuario_id, periodo, data_avaliacao, estado, total_20, observacoes, plano_melhoria, avaliado_por, avaliador_id) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";
            }

            if (!$errors) {
                $stmt = $conn->prepare($query);
                if ($isUpdate) {
                    $stmt->bind_param('isssdsssii', $usuario_id, $periodo, $data_avaliacao, $estado, $total20, $observacoes, $plano, $nomeFuncionario, $avaliadorId, $avaliacao_id);
                } else {
                    $stmt->bind_param('isssdsssi', $usuario_id, $periodo, $data_avaliacao, $estado, $total20, $observacoes, $plano, $nomeFuncionario, $avaliadorId);
                }
                $ok = $stmt->execute();
                $novoId = $isUpdate ? $avaliacao_id : $stmt->insert_id;
                $stmt->close();

                if ($ok && $novoId) {
                    // Limpar e reinserir detalhes
                    $conn->query("DELETE FROM avaliacao_detalhes WHERE avaliacao_id=" . (int)$novoId);
                    $ins = $conn->prepare("INSERT INTO avaliacao_detalhes (avaliacao_id, criterio_id, nota5, comentario) VALUES (?, ?, ?, ?)");
                    foreach ($criterios as $c) {
                        $cid = (int)$c['id'];
                        $nota5 = isset($itens[$cid]['nota5']) ? (float)$itens[$cid]['nota5'] : 0.0;
                        $coment = isset($itens[$cid]['comentario']) ? trim($itens[$cid]['comentario']) : null;
                        $ins->bind_param('iids', $novoId, $cid, $nota5, $coment);
                        $ins->execute();
                    }
                    $ins->close();
                    $okMsg = ($estado === 'FINAL') ? 'Avaliação finalizada.' : 'Rascunho salvo.';
                    if (!$isUpdate) $_GET['id'] = $novoId;
                } else {
                    $errors[] = 'Falha ao salvar a avaliação.';
                }
            }
        }
    }
}

// ======= CARREGAR AVALIAÇÃO SELECIONADA =======
$avaliacao = null;
$detalhes = [];
$nomeAvaliado = 'N/A';
$nomeAvaliador = 'N/A';
if (isset($_GET['id']) && ctype_digit($_GET['id'])) {
    $id = (int)$_GET['id'];
    $st = $conn->prepare("SELECT * FROM avaliacoes WHERE id=?");
    $st->bind_param('i', $id);
    $st->execute();
    $avaliacao = $st->get_result()->fetch_assoc();
    $st->close();
    if ($avaliacao) {
        $rows = all($conn, "SELECT * FROM avaliacao_detalhes WHERE avaliacao_id=" . (int)$id);
        foreach ($rows as $r) {
            $detalhes[(int)$r['criterio_id']] = $r;
        }
        $nomeAvaliado = getUserNameById($usuarios, $avaliacao['usuario_id']);
        $nomeAvaliador = getUserNameById($usuarios, $avaliacao['avaliador_id']);
    } // Chave de fechamento do 'if ($avaliacao)' adicionada aqui
}
// --- COLE A LÓGICA DE AVALIAÇÃO DE PLANOS DE AULA AQUI ---
/**
 * Função auxiliar para obter datas de início e fim de um trimestre
 * @param string $periodoStr Ex.: "1º Trimestre 2025"
 * @return array|null
 */
// ...

function getTrimestreDates($periodoStr) {
    if (preg_match('/(\d+)[º]? Trimestre (\d{4})/', $periodoStr, $matches)) {
        $trimestre = (int)$matches[1];
        $ano = (int)$matches[2];
        
        switch ($trimestre) {
            case 1: return ['start' => "{$ano}-01-01", 'end' => "{$ano}-03-31"];
            case 2: return ['start' => "{$ano}-04-01", 'end' => "{$ano}-06-30"];
            case 3: return ['start' => "{$ano}-07-01", 'end' => "{$ano}-09-30"];
            default: return null;
        }
    } // Adicione esta chave
    return null;
}


// 1. Encontra o ID do critério de planos de aula
$idPlanos = null;
foreach ($criterios as $c) {
    if ($c['nome'] === 'Pontualidade na Entrega de Planos de Aula') {
        $idPlanos = (int)$c['id'];
        break;
    }
}

// 2. Verifica se um funcionário e período estão definidos e se o critério existe
if ($idPlanos !== null && $uCur > 0 && !empty($perCur)) {
    $datas = getTrimestreDates($perCur);
    if ($datas) {
        $dataInicio = $datas['start'];
        $dataFim = $datas['end'];
        
        // 3. Conta os meses com planos enviados
        $sql = "SELECT COUNT(DISTINCT MONTH(data_upload)) FROM upload WHERE usuario_id = ? AND data_upload BETWEEN ? AND ?";
        $stmt = $conn->prepare($sql);
        if ($stmt) {
            $stmt->bind_param('iss', $uCur, $dataInicio, $dataFim);
            $stmt->execute();
            $result = $stmt->get_result();
            $mesesComPlanos = (int)$result->fetch_row()[0];
            $stmt->close();
            
            // 4. Calcula a nota de 0 a 5 de forma proporcional
            $totalMeses = 3;
            $nota5 = ($totalMeses > 0) ? ($mesesComPlanos / $totalMeses) * 5 : 0;
            
            // 5. Insere ou atualiza o array de detalhes para exibir a nota no formulário
            if (!($avaliacao && $avaliacao['estado'] === 'FINAL')) {
                $detalhes[$idPlanos] = [
                    'nota5' => round($nota5, 1),
                    'comentario' => "Avaliação automática de planos de aula: {$mesesComPlanos} de {$totalMeses} meses com envios no período."
                ];
            }
        }
    }
}





// ======= HISTÓRICO (filtros) =======
$f_user = (int)($_POST['f_user'] ?? 0);
$f_estado = $_POST['f_estado'] ?? '';
$f_periodo = $_POST['f_periodo'] ?? '';
$w = [];
if ($f_user) $w[] = 'a.usuario_id=' . $f_user;
if ($f_estado !== '') $w[] = "a.estado='" . $conn->real_escape_string($f_estado) . "'";
if ($f_periodo !== '') $w[] = "a.periodo='" . $conn->real_escape_string($f_periodo) . "'";
$where = $w ? ('WHERE ' . implode(' AND ', $w)) : '';
$historico = all($conn,
    "SELECT a.*, u.nome AS funcionario, u.perfil FROM avaliacoes a JOIN usuarios u ON u.id=a.usuario_id $where ORDER BY a.created_at DESC LIMIT 120"
);

?>
<!DOCTYPE html>
<html lang="pt">
<head>
<meta charset="utf-8" />
<meta name="viewport" content="width=device-width, initial-scale=1" />
<title>Painel da Direção — Avaliação de Funcionários</title>
<style>
/* CSS do tema escuro original */
:root{
    --bg:#0b1220;--panel:#0f172a;--card:#111827;--muted:#94a3b8;--text:#e5e7eb;
    --pri:#2563eb;--pri2:#1d4ed8;--ok:#16a34a;--warn:#d97706;--err:#dc2626;
    --border-dark:#1f2937;--border-light:#253049;
}
/* Novas variáveis para o tema claro */
body.light-mode{
    --bg:#f1f5f9;--panel:#e2e8f0;--card:#fff;--muted:#64748b;--text:#1e293b;
    --pri:#3b82f6;--pri2:#2563eb;--ok:#16a34a;--warn:#f59e0b;--err:#ef4444;
    --border-dark:#d1d5db;--border-light:#e5e7eb;
}
*{box-sizing:border-box}
body{
    margin:0;
    background:linear-gradient(180deg,var(--bg),#0a1020);
    color:var(--text);font-family:Inter,system-ui,Segoe UI,Arial,sans-serif;
    transition:background-color .3s, color .3s;
}
body.light-mode{
    background:linear-gradient(180deg,var(--bg),#e2e8f0);
}
.header{position:sticky;top:0;background:rgba(15,23,42,.85);backdrop-filter:blur(8px);border-bottom:1px solid var(--border-dark);z-index:10;transition:background .3s}
body.light-mode .header{background:rgba(226,232,240,.85)}
.container{max-width:1200px;margin:0 auto;padding:16px}
.hflex{display:flex;align-items:center;gap:12px}
.logo{font-weight:800}
.badge{background:var(--pri);color:#fff;border-radius:999px;padding:6px 10px;font-size:12px}
.grid{display:grid;gap:16px}
@media(min-width:900px){.grid{grid-template-columns:1.2fr .8fr}}
.card{background:linear-gradient(180deg,rgba(17,24,39,.9),rgba(17,24,39,.7));border:1px solid var(--border-dark);border-radius:18px;padding:16px;box-shadow:0 10px 25px rgba(0,0,0,.25);transition:background .3s, border-color .3s}
body.light-mode .card{background:var(--card);box-shadow:0 10px 25px rgba(0,0,0,.05)}
.label{font-size:12px;color:var(--muted);margin-bottom:6px}
.input,select,textarea{width:100%;background:#0b1220;border:1px solid var(--border-light);color:var(--text);border-radius:12px;padding:10px 12px;outline:none;transition:background .3s, border-color .3s, color .3s}
body.light-mode .input,body.light-mode select,body.light-mode textarea{background:var(--bg);border-color:var(--border-dark);color:var(--text)}
.row{display:grid;grid-template-columns:1fr 1fr;gap:12px}
@media(max-width:720px){.row{grid-template-columns:1fr}}
.kpi{display:flex;gap:10px;flex-wrap:wrap}
.pill{background:var(--bg);border:1px solid var(--border-light);border-radius:12px;padding:8px 12px;transition:background .3s}
.btns{display:flex;gap:8px;flex-wrap:wrap}
.btn{background:var(--pri);border:none;color:var(--text);padding:10px 14px;border-radius:12px;cursor:pointer;font-weight:700;transition:background .2s, color .2s}
.btn:hover{background:var(--pri2)}
.btn.outline{background:transparent;border:1px solid #334155;color:var(--text)}
body.light-mode .btn.outline{border-color:#9ca3af;color:var(--text)}
.btn.ok{background:var(--ok)}
.btn.warn{background:var(--warn)}
.btn.err{background:var(--err)}
.table{width:100%;border-collapse:collapse}
.table th,.table td{padding:10px;border-bottom:1px solid var(--border-light);text-align:left}
.table th{color:var(--pri)}
.state{padding:2px 8px;border-radius:999px;font-size:12px;display:inline-block}
.state.RASCUNHO{background:#1f2937;color:#fbbf24;border:1px solid #92400e}
body.light-mode .state.RASCUNHO{background:#fff;color:#f59e0b;border:1px solid #f59e0b}
.state.FINAL{background:#052e16;color:#86efac;border:1px solid #166534}
body.light-mode .state.FINAL{background:#fff;color:#16a34a;border:1px solid #16a34a}
.score{font-size:28px;font-weight:800}
.crit{display:grid;grid-template-columns:1.2fr .6fr;gap:10px;align-items:center;border:1px solid var(--border-light);border-radius:14px;padding:12px;margin-bottom:10px;transition:border-color .3s}
.slider{width:100%}
.small{font-size:12px;color:var(--muted)}
.print-hide{display:block}
.print-only{display:none}
.theme-switch{
    background:transparent;border:none;cursor:pointer;font-size:1.5em;color:var(--text);
    transition:color .2s;
}
.theme-switch:hover{color:var(--pri)}

/* CSS para Impressão */
@media print {
    .print-hide{display:none}
    .print-only{display:block;color:#000;background:#fff;padding:0;margin:0}
    body{background:#fff;color:#000;padding:0;margin:0;font-family:serif}
    .header,.card,aside,.logo,.badge{box-shadow:none;border:none}
    .print-grid{display:grid;gap:20px;grid-template-columns:1fr}
    h1,h2,h3{page-break-after:avoid}
    .print-signatures{display:flex;justify-content:space-around;margin-top:40px;text-align:center}
    .print-signatures div{width:45%;padding-top:20px;border-top:1px solid #000}
    .print-table{width:100%;border-collapse:collapse;margin-top:15px}
    .print-table th, .print-table td{padding:8px;border:1px solid #ccc;text-align:left;font-size:14px}
    .print-table th{background:#f0f0f0;font-weight:bold}
    .print-row{border-bottom:1px solid #e0e0e0;padding:10px 0}
}
</style>
</head>
<body>
<header class="header print-hide">
    <div class="container hflex">
        <div class="logo">SISMAG · Avaliação de Funcionários (Direção)</div>
        <span class="badge">Direção</span>
        <div style="margin-left:auto" class="small">Utilizador: <?= h($nomeUsuario) ?></div>
        <button id="theme-toggle" class="theme-switch" title="Alternar tema">
            🌙
        </button>
    </div>
</header>

<div class="container grid print-hide">
    <section class="card">
        <h3>Inserção / Edição de Avaliação</h3>
        <?php if ($errors): ?>
            <div class="small" style="color:#fecaca">• <?= implode('<br>• ', array_map('h', $errors)) ?></div>
        <?php endif; ?>
        <?php if ($okMsg): ?>
            <div class="small" style="color:#bbf7d0">✔ <?= h($okMsg) ?></div>
        <?php endif; ?>

        <?php
        $locked = ($avaliacao && $avaliacao['estado'] === 'FINAL');
        $curId = $avaliacao['id'] ?? ($_POST['avaliacao_id'] ?? '');
        $uCur = $avaliacao['usuario_id'] ?? (int)($_POST['usuario_id'] ?? 0);
        $perCur = $avaliacao['periodo'] ?? ($_POST['periodo'] ?? '');
        $dtCur = $avaliacao['data_avaliacao'] ?? ($_POST['data_avaliacao'] ?? date('Y-m-d'));
        $obsCur = $avaliacao['observacoes'] ?? ($_POST['observacoes'] ?? '');
        $planCur = $avaliacao['plano_melhoria'] ?? ($_POST['plano_melhoria'] ?? '');
        ?>

        <form method="post" id="form-av">
            <input type="hidden" name="csrf_token" value="<?= h($csrf) ?>" />
            <input type="hidden" name="avaliacao_id" value="<?= h($curId) ?>" />

            <div class="row">
                <div>
                    <div class="label">Funcionário (tabela `usuarios`)</div>
                    <select name="usuario_id" <?= $locked ? 'disabled' : '' ?>>
                        <option value="">— selecione —</option>
                        <?php foreach ($usuarios as $u): ?>
                            <option value="<?= $u['id'] ?>" <?= ($uCur == (int)$u['id']) ? 'selected' : '' ?>><?= h($u['nome']) ?> (<?= h($u['perfil']) ?>)</option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div>
                    <div class="label">Período</div>
                    <input class="input" name="periodo" placeholder="Ex.: 1º Trimestre 2025" value="<?= h($perCur) ?>" <?= $locked ? 'readonly' : '' ?> />
                </div>
            </div>

            <div class="row">
                <div>
                    <div class="label">Data da Avaliação</div>
                    <input class="input" type="date" name="data_avaliacao" value="<?= h($dtCur) ?>" <?= $locked ? 'readonly' : '' ?> />
                </div>
                <div>
                    <div class="label">Avaliador</div>
                    <input class="input" value="<?= h($nomeUsuario) ?>" readonly />
                </div>
            </div>

            <hr style="border-color:var(--border-light);border-width:1px 0 0;margin:14px 0;" />
            <div class="kpi">
                <div class="pill">Pontuação total (0–20): <span id="score" class="score">0.00</span></div>
                <div class="pill">Estado: <span class="state <?= h($avaliacao['estado'] ?? 'RASCUNHO') ?>"><?= h($avaliacao['estado'] ?? 'RASCUNHO') ?></span></div>
                <div class="pill">Pesos: <span id="pesosTot">0</span></div>
            </div>

            <h4 class="small" style="margin-top:10px">Para cada critério (peso indicado), atribua nota de 0 (muito fraco) a 5 (excelente). O total converte para escala 0–20.</h4>

            <?php
            $catAtual = '';
            foreach ($criterios as $c):
                if ($catAtual !== $c['categoria']) {
                    if ($catAtual !== '') echo '<br />';
                    echo '<h3>' . h($c['categoria']) . '</h3>';
                    $catAtual = $c['categoria'];
                }
                $cid = (int)$c['id'];
                $notaPre = 0.0;
                $comentPre = '';
                if (isset($detalhes[$cid])) {
                    $notaPre = (float)$detalhes[$cid]['nota5'];
                    $comentPre = (string)$detalhes[$cid]['comentario'];
                }
                if (isset($_POST['itens'][$cid])) {
                    $notaPre = (float)$_POST['itens'][$cid]['nota5'];
                    $comentPre = (string)$_POST['itens'][$cid]['comentario'];
                }
                ?>
                <div class="crit">
                    <div>
                        <div style="font-weight:700"><?= h($c['nome']) ?> <span class="small">(peso <?= h($c['peso']) ?>)</span></div>
                        <div class="small"><?= h($c['descricao']) ?></div>
                        <textarea name="itens[<?= $cid ?>][comentario]" rows="2" placeholder="Comentário (opcional): evidências, pontos fortes, melhorias" <?= $locked ? 'readonly' : '' ?>><?= h($comentPre) ?></textarea>
                    </div>
                    <div>
                        <input class="slider" type="range" min="0" max="5" step="0.5" value="<?= h($notaPre) ?>" name="itens[<?= $cid ?>][nota5]" data-peso="<?= h($c['peso']) ?>" <?= $locked ? 'disabled' : '' ?> />
                        <div class="pill" style="margin-top:8px"><span class="small">Nota:</span> <strong class="nota-val"><?= number_format($notaPre, 1) ?></strong></div>
                    </div>
                </div>
            <?php endforeach; ?>

            <div class="row">
                <div>
                    <div class="label">Observações gerais</div>
                    <textarea name="observacoes" rows="3" <?= $locked ? 'readonly' : '' ?>><?= h($obsCur) ?></textarea>
                </div>
                <div>
                    <div class="label">Plano de melhoria</div>
                    <textarea name="plano_melhoria" rows="3" <?= $locked ? 'readonly' : '' ?>><?= h($planCur) ?></textarea>
                </div>
            </div>

            <div class="btns print-hide" style="margin-top:10px">
                <?php if (!$locked): ?>
                    <button class="btn" type="submit" name="acao" value="salvar">💾 Salvar rascunho</button>
                    <button class="btn ok" type="submit" name="acao" value="finalizar" onclick="return confirm('Finalizar avaliação? Não poderá editar depois.')">✅ Finalizar</button>
                <?php endif; ?>
                <a class="btn outline" href="?">➕ Nova</a>
                <?php if ($curId): ?><a class="btn outline" href="?id=<?= (int)$curId ?>">🔄 Recarregar</a><button class="btn warn" type="button" onclick="window.print()">🖨 Imprimir</button><?php endif; ?>
            </div>
        </form>
    </section>

    <aside class="card">
        <h3>Histórico recente</h3>
        <form method="post" class="print-hide" style="margin-bottom:10px">
            <input type="hidden" name="csrf_token" value="<?= h($csrf) ?>" />
            <div class="row">
                <div>
                    <div class="label">Funcionário</div>
                    <select name="f_user">
                        <option value="0">— todos —</option>
                        <?php foreach ($usuarios as $u): ?>
                            <option value="<?= $u['id'] ?>" <?= ($f_user == (int)$u['id']) ? 'selected' : '' ?>><?= h($u['nome']) ?> (<?= h($u['perfil']) ?>)</option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div>
                    <div class="label">Estado</div>
                    <select name="f_estado">
                        <option value="">— todos —</option>
                        <option value="RASCUNHO" <?= ($f_estado === 'RASCUNHO') ? 'selected' : '' ?>>Rascunho</option>
                        <option value="FINAL" <?= ($f_estado === 'FINAL') ? 'selected' : '' ?>>Final</option>
                    </select>
                </div>
            </div>
            <div class="row">
                <div>
                    <div class="label">Período</div>
                    <input class="input" name="f_periodo" placeholder="Ex.: 1º Trimestre 2025" value="<?= h($f_periodo) ?>" />
                </div>
                <div class="btns" style="align-items:end">
                    <button class="btn" type="submit">Filtrar</button>
                    <button class="btn outline" type="submit" name="acao" value="export_csv">⬇ Exportar CSV</button>
                </div>
            </div>
        </form>

        <div style="max-height:520px;overflow:auto">
            <table class="table">
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Funcionário</th>
                        <th>Perfil</th>
                        <th>Período</th>
                        <th>Data</th>
                        <th>Estado</th>
                        <th>Total</th>
                        <th></th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($historico as $h): ?>
                        <tr>
                            <td><?= (int)$h['id'] ?></td>
                            <td><?= h($h['funcionario']) ?></td>
                            <td><?= h($h['perfil']) ?></td>
                            <td><?= h($h['periodo']) ?></td>
                            <td><?= h($h['data_avaliacao']) ?></td>
                            <td><span class="state <?= h($h['estado']) ?>"><?= h($h['estado']) ?></span></td>
                            <td><?= number_format((float)$h['total_20'], 2) ?></td>
                            <td><a class="btn outline" href="?id=<?= (int)$h['id'] ?>">Abrir</a></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </aside>
</div>

<div class="print-only">
    <div style="text-align:center;margin-bottom:20px;padding-top:20px;">
        <h2>Avaliação de Desempenho do Funcionário</h2>
        <p><strong>Funcionário:</strong> <?= h($nomeAvaliado) ?></p>
        <p><strong>Período:</strong> <?= h($avaliacao['periodo'] ?? 'N/A') ?></p>
        <p><strong>Data da Avaliação:</strong> <?= h($avaliacao['data_avaliacao'] ?? 'N/A') ?></p>
        <p><strong>Avaliador:</strong> <?= h($nomeAvaliador) ?></p>
    </div>

    <div style="padding:0 20px;">
        <h3>Pontuação Final: <span style="font-size:24px;color:green;"><?= number_format((float)($avaliacao['total_20'] ?? 0), 2) ?></span> / 20</h3>

        <table class="print-table">
            <thead>
                <tr>
                    <th>Critério de Avaliação</th>
                    <th>Peso</th>
                    <th>Nota (0-5)</th>
                    <th>Comentário</th>
                </tr>
            </thead>
            <tbody>
                <?php
                $catAtual = '';
                foreach ($criterios as $c):
                    if ($catAtual !== $c['categoria']) {
                        echo '<tr><td colspan="4" style="font-weight:bold;background:#f0f0f0;">' . h($c['categoria']) . '</td></tr>';
                        $catAtual = $c['categoria'];
                    }
                    $cid = (int)$c['id'];
                    $notaPre = isset($detalhes[$cid]) ? (float)$detalhes[$cid]['nota5'] : 0.0;
                    $comentPre = isset($detalhes[$cid]) ? (string)$detalhes[$cid]['comentario'] : 'N/A';
                ?>
                <tr>
                    <td><?= h($c['nome']) ?></td>
                    <td><?= number_format((float)$c['peso'], 2) ?></td>
                    <td><?= number_format($notaPre, 1) ?></td>
                    <td><?= h($comentPre) ?></td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>

        <br>
        <p><strong>Observações Gerais:</strong><br><?= nl2br(h($avaliacao['observacoes'] ?? 'N/A')) ?></p>
        <p><strong>Plano de Melhoria:</strong><br><?= nl2br(h($avaliacao['plano_melhoria'] ?? 'N/A')) ?></p>
    </div>

    <div class="print-signatures">
        <div>
            <p>_____________________________________</p>
            <p>Assinatura do Funcionário Avaliado</p>
            <p>Nome: <?= h($nomeAvaliado) ?></p>
        </div>
        <div>
            <p>_____________________________________</p>
            <p>Assinatura do Avaliador</p>
            <p>Nome: <?= h($nomeAvaliador) ?></p>
        </div>
    </div>
</div>

<script>
    (function() {
        // Lógica de alternar tema
        const themeToggle = document.getElementById('theme-toggle');
        const body = document.body;
        
        function setTheme(mode) {
            if (mode === 'light') {
                body.classList.add('light-mode');
                themeToggle.textContent = '☀️';
                themeToggle.title = 'Alternar para tema escuro';
            } else {
                body.classList.remove('light-mode');
                themeToggle.textContent = '🌙';
                themeToggle.title = 'Alternar para tema claro';
            }
            localStorage.setItem('theme', mode);
        }

        const currentTheme = localStorage.getItem('theme') || 'dark';
        setTheme(currentTheme);

        themeToggle.addEventListener('click', () => {
            const newTheme = body.classList.contains('light-mode') ? 'dark' : 'light';
            setTheme(newTheme);
        });

        // Lógica do formulário (mantida do código anterior)
        const form = document.getElementById('form-av');
        if (!form) return;
        const sliders = form.querySelectorAll('input[type="range"]');
        const scoreEl = document.getElementById('score');
        const pesosTot = document.getElementById('pesosTot');
        const locked = <?= json_encode($locked ?? false) ?>;

        function recalc() {
            let sumW = 0,
                sumWN = 0;
            sliders.forEach(sl => {
                const w = parseFloat(sl.dataset.peso || '1');
                const n = parseFloat(sl.value || '0'); // 0–5
                sumW += w;
                sumWN += w * n;
                const v = sl.closest('.crit').querySelector('.nota-val');
                if (v) v.textContent = n.toFixed(1);
            });
            pesosTot.textContent = sumW.toFixed(2);
            const total20 = sumW > 0 ? (sumWN / sumW) * 4 : 0;
            scoreEl.textContent = total20.toFixed(2);
        }
        recalc();
        sliders.forEach(sl => sl.addEventListener('input', recalc));

        if (!locked) {
            let t = null;
            function schedule() {
                if (t) clearTimeout(t);
                t = setTimeout(() => {
                    const ghost = document.createElement('input');
                    ghost.type = 'hidden';
                    ghost.name = 'acao';
                    ghost.value = 'salvar';
                    form.appendChild(ghost);
                    fetch(location.href, {
                        method: 'POST',
                        body: new FormData(form)
                    }).then(() => ghost.remove());
                }, 20000);
            }
            form.addEventListener('input', schedule);
        }

        const usuarioSelect = form.querySelector('select[name="usuario_id"]');
        form.addEventListener('submit', function(event) {
            if (usuarioSelect) {
                usuarioSelect.disabled = false;
            }
        });
    })();
</script>
</body>
</html>